/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;

import com.inet.report.Area;
import com.inet.report.Element;
import com.inet.report.Engine;
import com.inet.report.FieldElement;
import com.inet.report.RDC;
import com.inet.report.Section;

/**
 * This example shows how to load an existing report from file to get and change some existing report elements.<br>
 * To do this you have to get the elements vector of the section that holds the element thats looks like that you want
 * to change. You can do this in two ways. First you can use the method <code>getElementsV()</code> in class
 * <code>Section</code>. This returns a Vector with all the elements that are in the section, so if you add a element to
 * section it will appear in this vector, too. <br>
 * The second way is to use the method <code>getElements()</code> in class <code>Section</code>. This returns an
 * <code>Element[]</code> with all elements that were in the section at the moment you call this method.<br>
 * If you have one of the sets (Vector or Array) you can walk through with a loop and test the instance of the elements
 * into. Additionally you can test some properties like the field of a FieldElement.<br>
 * <br>
 * If you want to get all elements of the hole report you have to add one loop for the Areas (<code>getArea(int)</code>,
 * <code>getAreaCount()</code> in class <code>Engine</code>) and one loop for the sections in each area (
 * <code>getSection(i)</code>, <code>getSectionCount()</code> in class <code>Area</code>).
 */
public class GettingExistingElementsSample extends RDCSample {

    //The URL to the used report file.
    //Change this before running example if the location is another one.
    //Comes with this sample suite.
    static final String MAINREPORT = "samples/rdc/mainrep.rpt";

    /**
     * Change some properties of field elements in the detail section of an existing report
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //create a new Engine
            Engine eng = new Engine( exportFmt );
            //set the name of the report file, that contains the connection to the database
            URL reportURL = getClass().getResource( '/' + MAINREPORT );
            if( reportURL != null ) {
                eng.setReportFile( reportURL );
            } else {
                eng.setReportFile( MAINREPORT );
            }

            //change the lookout of the field elements in detail section
            Area dArea = eng.getArea( "D" );
            Section dSec = dArea.getSection( 0 );

            //for all elements
            for( Element elem : dSec.getElementsV() ) {
                //test the instance
                if( elem instanceof FieldElement ) {
                    FieldElement fElem = (FieldElement)elem;
                    fElem.setFontColor( RDC.COLOR_BLUE );//change color
                    fElem.setItalic( true );//set font cursive
                    fElem.setFontSize( 12 );//set font size to 12
                    //get one special FieldElement (that hold the field with name "Customer.Contact Last Name")
                    if( fElem.getField().getName().equals( "Customers.ContactName" ) ) {
                        fElem.setHeight( fElem.getHeight() + 100 );//height += 100;
                        fElem.setFontSize( 14 );//set font size to 14
                        fElem.setUnderline( true );//set font underlined
                    }
                }
            }
            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new GettingExistingElementsSample().initUI();
    }
}
